/*
 * Decompiled with CFR 0.152.
 */
package com.ssl.code.signing.tool.commands;

import com.ssl.code.signing.tool.code.CatSignature;
import com.ssl.code.signing.tool.code.CodeSignature;
import com.ssl.code.signing.tool.code.JarSignature;
import com.ssl.code.signing.tool.code.OvaSignature;
import com.ssl.code.signing.tool.csc.CscApi;
import com.ssl.code.signing.tool.exception.CscApiException;
import com.ssl.code.signing.tool.file.OvfFile;
import com.ssl.code.signing.tool.mscab.MSCabinetFile;
import com.ssl.code.signing.tool.util.Paths;
import de.taimos.totp.TOTP;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Properties;
import java.util.stream.Stream;
import net.jsign.DigestAlgorithm;
import net.jsign.msi.MSIFile;
import net.jsign.pe.PEFile;
import net.jsign.script.JScript;
import net.jsign.script.PowerShellScript;
import net.jsign.script.PowerShellXMLScript;
import net.jsign.script.VBScript;
import net.jsign.script.WindowsScript;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.core.config.ConfigurationSource;
import org.apache.logging.log4j.core.config.Configurator;
import picocli.CommandLine;

@CommandLine.Command(name="batch_sign_hash", description={"Batch sign of pre-computed hash(es)"}, version={"v1.2.7"}, mixinStandardHelpOptions=true)
public class BatchSignHashCommand
implements Runnable {
    private static final Logger logger;
    @CommandLine.Option(names={"-access_token"}, description={"Access token"}, required=true)
    private String accessToken;
    @CommandLine.Option(names={"-input_dir_path"}, description={"Input directory path where to pick unsigned files and correspondinghash files for signing"}, required=true)
    private String inputDirPath;
    @CommandLine.Option(names={"-output_dir_path"}, description={"Output directory to place signed files"}, required=true)
    private String outputDirPath;
    @CommandLine.Option(names={"-otp"}, description={"OTP value"}, required=false)
    private String otp;
    @CommandLine.Option(names={"-credential_id"}, description={"Credential ID"}, required=false)
    private String credentialId;
    @CommandLine.Option(names={"-totp_secret"}, description={"TOTP secret"}, required=false)
    private String totpSecret;

    @Override
    public void run() {
        try {
            logger.info("Command: Batch sign hash");
            Properties props = new Properties();
            props.load(new FileInputStream(Paths.getConfigFilePath()));
            String clientId = props.getProperty("CLIENT_ID");
            logger.info("Client ID: " + clientId);
            String oauth2Endpoint = props.getProperty("OAUTH2_ENDPOINT");
            logger.info("OAuth2 URL: " + oauth2Endpoint);
            String cscEndpoint = props.getProperty("CSC_API_ENDPOINT");
            logger.info("CSC Endpoint: " + cscEndpoint);
            String tsaUrl = props.getProperty("TSA_URL");
            logger.info("TSA URL: " + tsaUrl);
            File inputDirFile = new File(this.inputDirPath);
            if (!inputDirFile.exists()) {
                logger.info("Not a valid input directory: " + this.inputDirPath);
                System.out.print("Error: Not a valid input directory - " + this.inputDirPath);
                return;
            }
            File outputDirFile = new File(this.outputDirPath);
            if (!outputDirFile.exists()) {
                logger.info("Not a valid output directory: " + this.outputDirPath);
                System.out.print("Error: Not a valid output directory - " + this.outputDirPath);
                return;
            }
            HashMap<String, String> filesInfo = this.validateInputDirectory(inputDirFile);
            if (filesInfo == null) {
                return;
            }
            CscApi cscApi = new CscApi(this.accessToken, cscEndpoint);
            String[] evcsCredentialIds = null;
            String[] ovcsCredentialIds = null;
            String[] credentialIds = null;
            if (this.credentialId != null && !this.credentialId.equals("")) {
                logger.info("Credential ID in input parameter: " + this.credentialId);
                credentialIds = new String[]{this.credentialId};
            } else {
                logger.info("No credential ID provided in input parameter");
                evcsCredentialIds = cscApi.getCredentialIDs("EVCS");
                ovcsCredentialIds = cscApi.getCredentialIDs("OVCS");
                credentialIds = (String[])Stream.concat(Arrays.stream(evcsCredentialIds), Arrays.stream(ovcsCredentialIds)).toArray(String[]::new);
            }
            if (credentialIds.length == 0) {
                logger.info("No credential ID found for this user");
                System.out.print("Error: No credential ID found for this user");
                return;
            }
            if (credentialIds.length == 1) {
                this.credentialId = credentialIds[0];
                logger.info("Credential ID retrieved from signatures service: " + this.credentialId);
                List<X509Certificate> certs = cscApi.getCredentialInfo(credentialIds[0]);
                logger.info("Subject DN: " + certs.get(0).getSubjectDN());
                logger.info("Issuer DN: " + certs.get(0).getIssuerDN());
                logger.info("Certificate expiry: " + certs.get(0).getNotAfter());
                ArrayList<String> hashesList = new ArrayList<String>();
                for (String hashFileName : filesInfo.keySet()) {
                    File file = new File(this.inputDirPath + File.separator + hashFileName);
                    byte[] hash = Files.readAllBytes(file.toPath());
                    hashesList.add(new String(hash));
                }
                String[] hashes = new String[hashesList.size()];
                hashesList.toArray(hashes);
                if (this.otp != null && this.totpSecret != null) {
                    System.out.print("Error: Only one value from OTP or TOTP secret needs to be provided");
                    return;
                }
                if (this.otp == null && this.totpSecret == null) {
                    System.out.print("Error: Either OTP or TOTP secret value must be provided");
                    return;
                }
                if (this.totpSecret != null && !this.totpSecret.equals("")) {
                    byte[] totpSeed = Base64.getDecoder().decode(this.totpSecret);
                    String hexKey = Hex.encodeHexString((byte[])totpSeed);
                    this.otp = TOTP.getOTP((String)hexKey);
                }
                String sad = cscApi.getCredentialsAuthorize(this.credentialId, hashes.length, hashes, this.otp);
                String[] signatures = cscApi.signHash(this.credentialId, sad, hashes, "1.2.840.113549.1.1.11", false);
                int i = 0;
                for (String hashFileName : filesInfo.keySet()) {
                    String unsignedFileName = filesInfo.get(hashFileName);
                    File srcFile = new File(this.inputDirPath + File.separator + unsignedFileName);
                    File targetFile = new File(this.outputDirPath + File.separator + unsignedFileName);
                    FileUtils.copyFile((File)srcFile, (File)targetFile);
                    String ext = FilenameUtils.getExtension((String)unsignedFileName);
                    String codeFormatType = this.getCodeFormatType(ext);
                    File progNameFile = new File(this.inputDirPath + File.separator + "program_name.txt");
                    String programName = null;
                    if (progNameFile.exists()) {
                        programName = new String(Files.readAllBytes(progNameFile.toPath()));
                    }
                    this.sign(targetFile, codeFormatType, signatures[i], certs, tsaUrl, programName, srcFile.getPath());
                    ++i;
                }
            } else {
                System.out.println("Multiple valid credential IDs are available for this user. Please use the '-credential_id' option to designate which certificate to use for signing. Please use the 'credential_info' command to obtain more details about each credential ID");
                if (evcsCredentialIds.length != 0) {
                    System.out.println("EVCS Credential ID(s): ");
                    for (String credId : evcsCredentialIds) {
                        logger.info("Credential ID: " + credId);
                        System.out.println("- " + credId);
                    }
                }
                if (ovcsCredentialIds.length != 0) {
                    System.out.println("OVCS Credential ID(s): ");
                    for (String credId : ovcsCredentialIds) {
                        logger.info("Credential ID: " + credId);
                        System.out.println("- " + credId);
                    }
                }
                return;
            }
            System.out.println("Batch sign hash command executed successfully. Output directory for signed files: " + this.outputDirPath);
        }
        catch (CscApiException ex) {
            logger.error("", (Throwable)ex);
            System.out.println("Error: " + ex.getMessage());
        }
        catch (Exception ex) {
            ex.printStackTrace();
            logger.error("", (Throwable)ex);
        }
    }

    private HashMap<String, String> validateInputDirectory(File inputDir) throws Exception {
        String fileExtension;
        boolean programFilePresent = false;
        LinkedHashMap<String, String> filesInfo = new LinkedHashMap<String, String>();
        File[] listFiles = inputDir.listFiles(OvfFile.ignoreFiles());
        if (listFiles.length > 201) {
            logger.info("Error: only 100 files can be signed in a single batch");
            System.out.print("Error: only 100 files can be signed in a single batch");
            return null;
        }
        for (File file : listFiles) {
            if (file.isDirectory()) {
                logger.info("Error: Directory must not be present: " + file.getName());
                System.out.print("Error: Directory must not be present: " + file.getName());
                return null;
            }
            if (!file.isFile() || !(fileExtension = FilenameUtils.getExtension((String)file.getName())).equals("hash")) continue;
            filesInfo.put(file.getName(), "CODE");
        }
        for (File file : listFiles) {
            if (file.getName().equals("program_name.txt")) {
                programFilePresent = true;
                continue;
            }
            fileExtension = FilenameUtils.getExtension((String)file.getName());
            if (fileExtension.equals("hash")) continue;
            if (!(fileExtension.equals("acm") || fileExtension.equals("ax") || fileExtension.equals("cpl") || fileExtension.equals("dll") || fileExtension.equals("drv") || fileExtension.equals("efi") || fileExtension.equals("exe") || fileExtension.equals("mui") || fileExtension.equals("ocx") || fileExtension.equals("scr") || fileExtension.equals("sys") || fileExtension.equals("tsp") || fileExtension.equals("bin") || fileExtension.equals("msi") || fileExtension.equals("ps1") || fileExtension.equals("ps1xml") || fileExtension.equals("js") || fileExtension.equals("vbs") || fileExtension.equals("wsf") || fileExtension.equals("jar") || fileExtension.equals("cat") || fileExtension.equals("cab") || fileExtension.equals("ova") || fileExtension.equals("ovf"))) {
                logger.info("Error: File extension not allowed: " + file.getName());
                System.out.print("Error: File extension not allowed: " + file.getName());
                return null;
            }
            String usignedFilename = (String)((HashMap)filesInfo).get(file.getName() + ".hash");
            if (usignedFilename == null) {
                logger.info("Error: No matching hash file found against unsigned file: " + file.getName());
                System.out.print("Error: No matching hash file found against unsigned file: " + file.getName());
                return null;
            }
            filesInfo.put(file.getName() + ".hash", file.getName());
        }
        if (filesInfo.size() * 2 != listFiles.length) {
            if (programFilePresent) {
                if (filesInfo.size() * 2 + 1 != listFiles.length) {
                    logger.info("Error: remove extra files from input directory: " + inputDir.getAbsolutePath());
                    System.out.print("Error: remove extra files from input directory: " + inputDir.getAbsolutePath());
                    return null;
                }
            } else {
                logger.info("Error: remove extra files from input directory: " + inputDir.getAbsolutePath());
                System.out.print("Error: remove extra files from the input directory: " + inputDir.getAbsolutePath());
                return null;
            }
        }
        return filesInfo;
    }

    private void sign(File codeFilePath, String codeFormatType, String signature, List<X509Certificate> certsList, String tsaUrl, String programName, String inputFilePath) throws Exception {
        if (codeFormatType.equals("pe")) {
            PEFile file = new PEFile(codeFilePath);
            file.pad(8);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("msi")) {
            MSIFile file = new MSIFile(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("ps1")) {
            PowerShellScript file = new PowerShellScript(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("ps1xml")) {
            PowerShellXMLScript file = new PowerShellXMLScript(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("js")) {
            JScript file = new JScript(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("vbs")) {
            VBScript file = new VBScript(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("wsf")) {
            WindowsScript file = new WindowsScript(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("cab")) {
            MSCabinetFile file = new MSCabinetFile(codeFilePath);
            file.setSignature(CodeSignature.generateSignedData(file.createIndirectData(DigestAlgorithm.SHA256), signature, certsList, tsaUrl, programName));
            file.save();
        } else if (codeFormatType.equals("jar")) {
            X509Certificate[] certs = new X509Certificate[certsList.size()];
            JarSignature jarSignature = new JarSignature();
            byte[] signedJar = jarSignature.sign(signature, codeFilePath.getPath(), tsaUrl, certsList.toArray(certs));
            FileOutputStream fout = new FileOutputStream(codeFilePath);
            fout.write(signedJar);
            fout.close();
        } else if (codeFormatType.equals("cat")) {
            CatSignature catSignature = new CatSignature();
            byte[] signedCat = catSignature.sign(signature, codeFilePath, tsaUrl, certsList, programName);
            FileOutputStream fout = new FileOutputStream(codeFilePath);
            fout.write(signedCat);
            fout.close();
        } else if (codeFormatType.equals("ova")) {
            OvaSignature ovaSignature = new OvaSignature(inputFilePath, codeFormatType);
            byte[] signedOva = ovaSignature.sign(signature, certsList);
            FileOutputStream fout = new FileOutputStream(codeFilePath);
            fout.write(signedOva);
            fout.close();
        } else if (codeFormatType.equals("ovf")) {
            OvaSignature ovaSignature = new OvaSignature(codeFilePath.getPath(), codeFormatType);
            ovaSignature.copyFiles(inputFilePath);
            codeFilePath = new File(FilenameUtils.removeExtension((String)codeFilePath.getPath()) + ".cert");
            byte[] signedOva = ovaSignature.sign(signature, certsList);
            FileOutputStream fout = new FileOutputStream(codeFilePath);
            fout.write(signedOva);
            fout.close();
        }
    }

    private String getCodeFormatType(String fileExtension) {
        if (fileExtension.equals("acm") || fileExtension.equals("ax") || fileExtension.equals("cpl") || fileExtension.equals("dll") || fileExtension.equals("drv") || fileExtension.equals("efi") || fileExtension.equals("exe") || fileExtension.equals("mui") || fileExtension.equals("ocx") || fileExtension.equals("scr") || fileExtension.equals("sys") || fileExtension.equals("tsp") || fileExtension.equals("bin")) {
            return "pe";
        }
        if (fileExtension.equals("msi")) {
            return "msi";
        }
        if (fileExtension.equals("ps1")) {
            return "ps1";
        }
        if (fileExtension.equals("ps1xml")) {
            return "ps1xml";
        }
        if (fileExtension.equals("js")) {
            return "js";
        }
        if (fileExtension.equals("vbs")) {
            return "vbs";
        }
        if (fileExtension.equals("wsf")) {
            return "wsf";
        }
        if (fileExtension.equals("jar")) {
            return "jar";
        }
        if (fileExtension.equals("cat")) {
            return "cat";
        }
        if (fileExtension.equals("cab")) {
            return "cab";
        }
        if (fileExtension.equals("ova")) {
            return "ova";
        }
        if (fileExtension.equals("ovf")) {
            return "ovf";
        }
        return null;
    }

    static {
        try {
            FileInputStream inputStream = new FileInputStream(Paths.getLog4j2Path());
            ConfigurationSource source = new ConfigurationSource((InputStream)inputStream);
            Configurator.initialize(null, (ConfigurationSource)source);
        }
        catch (Exception ex) {
            ex.printStackTrace();
        }
        logger = LogManager.getLogger(BatchSignHashCommand.class);
    }
}

